/**
 * LinkReader - Background Service Worker (Manifest V3)
 * Receives extracted profile data from content.js and sends it to the API.
 * JWT token stored in chrome.storage.local (never localStorage).
 * API keys are NEVER stored client-side — all AI processing happens server-side.
 */

const API_BASE = "http://localhost:10042";

// ---------------------------------------------------------------------------
// Device Fingerprint (anti-abuse — survives VPN, new emails, incognito)
// ---------------------------------------------------------------------------

/**
 * Generate a stable device fingerprint from browser properties.
 * This hash survives VPN changes, new accounts, and extension reinstalls
 * because it's based on hardware/software traits, not IP or cookies.
 */
async function generateFingerprint() {
  const components = [];

  // Screen properties (hardware-tied)
  components.push(`${screen.width}x${screen.height}x${screen.colorDepth}`);
  components.push(screen.pixelDepth);

  // Timezone
  components.push(Intl.DateTimeFormat().resolvedOptions().timeZone);
  components.push(new Date().getTimezoneOffset());

  // Language and platform
  components.push(navigator.language);
  components.push(navigator.languages ? navigator.languages.join(",") : "");
  components.push(navigator.platform);
  components.push(navigator.hardwareConcurrency || 0);
  components.push(navigator.deviceMemory || 0);

  // User agent
  components.push(navigator.userAgent);

  // Hash the combined string
  const raw = components.join("|");
  const encoder = new TextEncoder();
  const data = encoder.encode(raw);
  const hashBuffer = await crypto.subtle.digest("SHA-256", data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  return hashArray.map((b) => b.toString(16).padStart(2, "0")).join("");
}

/**
 * Get or create the device ID (random UUID, persists in storage).
 * Combined with fingerprint for two-layer identification.
 */
async function getDeviceId() {
  const result = await chrome.storage.local.get("lr_device_id");
  if (result.lr_device_id) return result.lr_device_id;

  const deviceId = crypto.randomUUID();
  await chrome.storage.local.set({ lr_device_id: deviceId });
  return deviceId;
}

/**
 * Get the full device identity payload (sent with auth and extraction requests).
 */
async function getDeviceIdentity() {
  const [deviceId, fingerprint] = await Promise.all([
    getDeviceId(),
    generateFingerprint(),
  ]);
  return { device_id: deviceId, fingerprint: fingerprint };
}

// ---------------------------------------------------------------------------
// Auth helpers
// ---------------------------------------------------------------------------

/** Get the stored JWT token */
async function getToken() {
  const result = await chrome.storage.local.get("lr_token");
  return result.lr_token || null;
}

/** Store JWT token */
async function setToken(token) {
  await chrome.storage.local.set({ lr_token: token });
}

/** Remove stored token (logout) */
async function clearToken() {
  await chrome.storage.local.remove("lr_token");
}

/** Get stored user info */
async function getUserInfo() {
  const result = await chrome.storage.local.get("lr_user");
  return result.lr_user || null;
}

/** Store user info */
async function setUserInfo(user) {
  await chrome.storage.local.set({ lr_user: user });
}

// ---------------------------------------------------------------------------
// Recently extracted profiles (stored locally for popup display)
// ---------------------------------------------------------------------------

async function getRecentProfiles() {
  const result = await chrome.storage.local.get("lr_recent_profiles");
  return result.lr_recent_profiles || [];
}

async function addRecentProfile(profile) {
  const recent = await getRecentProfiles();
  // Add to front, keep last 20
  recent.unshift({
    name: profile.full_name,
    headline: profile.headline,
    linkedin_url: profile.linkedin_url,
    extracted_at: profile.extracted_at,
  });
  if (recent.length > 20) recent.length = 20;
  await chrome.storage.local.set({ lr_recent_profiles: recent });
}

// ---------------------------------------------------------------------------
// API communication
// ---------------------------------------------------------------------------

/**
 * Send extracted profile data to the API backend.
 * POST /api/profiles/extract with JWT auth.
 */
async function sendProfileToAPI(profileData) {
  const token = await getToken();
  if (!token) {
    return { success: false, error: "Not logged in. Please sign in first." };
  }

  try {
    // Include device identity for abuse prevention
    const device = await getDeviceIdentity();
    const payload = { ...profileData, ...device };

    const response = await fetch(`${API_BASE}/api/profiles/extract`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        Authorization: `Bearer ${token}`,
      },
      body: JSON.stringify(payload),
    });

    if (response.status === 401) {
      await clearToken();
      return { success: false, error: "Session expired. Please log in again." };
    }

    if (response.status === 429) {
      return {
        success: false,
        error: "Extraction limit reached. Upgrade your plan for more.",
      };
    }

    if (!response.ok) {
      const errBody = await response.json().catch(() => ({}));
      return {
        success: false,
        error: errBody.detail || `Server error (${response.status})`,
      };
    }

    const data = await response.json();
    // Save to recent profiles list
    await addRecentProfile(profileData);
    return { success: true, data };
  } catch (err) {
    return {
      success: false,
      error: "Cannot reach LinkReader server. Is it running?",
    };
  }
}

/**
 * Login to the API and store JWT token.
 */
async function login(email, password) {
  try {
    // Include device identity for abuse tracking
    const device = await getDeviceIdentity();

    const response = await fetch(`${API_BASE}/api/auth/login`, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ email, password, ...device }),
    });

    if (!response.ok) {
      const errBody = await response.json().catch(() => ({}));
      return {
        success: false,
        error: errBody.detail || "Invalid email or password",
      };
    }

    const data = await response.json();
    await setToken(data.access_token);
    await setUserInfo({ email, name: data.name || email });

    return { success: true, user: { email, name: data.name || email } };
  } catch (err) {
    return {
      success: false,
      error: "Cannot reach LinkReader server. Is it running?",
    };
  }
}

/**
 * Fetch usage/extraction count from API.
 */
async function getUsage() {
  const token = await getToken();
  if (!token) return { success: false, error: "Not logged in" };

  try {
    const response = await fetch(`${API_BASE}/api/auth/usage`, {
      headers: { Authorization: `Bearer ${token}` },
    });

    if (response.status === 401) {
      await clearToken();
      return { success: false, error: "Session expired" };
    }

    if (!response.ok) {
      return { success: false, error: "Failed to fetch usage" };
    }

    const data = await response.json();
    return { success: true, data };
  } catch (err) {
    return { success: false, error: "Cannot reach server" };
  }
}

// ---------------------------------------------------------------------------
// Message handling
// ---------------------------------------------------------------------------

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // Profile extracted from content script — send to API
  if (message.type === "profile_extracted") {
    sendProfileToAPI(message.data).then(sendResponse);
    return true; // async
  }

  // Login request from popup
  if (message.type === "login") {
    login(message.email, message.password).then(sendResponse);
    return true;
  }

  // Logout request from popup
  if (message.type === "logout") {
    (async () => {
      await clearToken();
      await chrome.storage.local.remove("lr_user");
      sendResponse({ success: true });
    })();
    return true;
  }

  // Check auth state
  if (message.type === "check_auth") {
    (async () => {
      const token = await getToken();
      const user = await getUserInfo();
      sendResponse({ loggedIn: !!token, user });
    })();
    return true;
  }

  // Get usage info
  if (message.type === "get_usage") {
    getUsage().then(sendResponse);
    return true;
  }

  // Get recent profiles
  if (message.type === "get_recent") {
    getRecentProfiles().then((profiles) => sendResponse({ profiles }));
    return true;
  }

  // Extraction progress (forwarded from content script to popup)
  if (message.type === "extraction_progress") {
    // Forward to any open popup — popup listens via onMessage
    // No response needed
    return false;
  }

  return false;
});

// ---------------------------------------------------------------------------
// Extension lifecycle
// ---------------------------------------------------------------------------

chrome.runtime.onInstalled.addListener(async (details) => {
  if (details.reason === "install") {
    console.log("LinkReader extension installed");
  } else if (details.reason === "update") {
    console.log(`LinkReader updated to v${chrome.runtime.getManifest().version}`);
  }

  // Re-inject content script into any already-open LinkedIn tabs
  // so the user doesn't have to manually refresh the page
  try {
    const tabs = await chrome.tabs.query({ url: ["https://www.linkedin.com/in/*", "https://linkedin.com/in/*"] });
    for (const tab of tabs) {
      chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ["content.js"],
      }).catch(() => {}); // Ignore tabs where injection fails
    }
  } catch (_) {}
});
