/**
 * LinkReader - Content Script
 * Runs on LinkedIn profile pages (/in/*).
 * When triggered by popup, expands all hidden sections then extracts structured profile data.
 * NEVER stores LinkedIn credentials. Only reads the current page DOM.
 */

(() => {
  "use strict";

  // Guard against double-injection (popup injects as fallback + manifest auto-injects)
  if (window.__linkReaderContentLoaded) return;
  window.__linkReaderContentLoaded = true;

  // ---------------------------------------------------------------------------
  // Utilities
  // ---------------------------------------------------------------------------

  /** Sleep helper */
  const sleep = (ms) => new Promise((r) => setTimeout(r, ms));

  /** Send progress updates back to popup via background */
  function reportProgress(step, detail) {
    chrome.runtime.sendMessage({
      type: "extraction_progress",
      step,
      detail,
    });
  }

  // ---------------------------------------------------------------------------
  // 1. Section Expansion
  // ---------------------------------------------------------------------------

  /**
   * Selectors / text patterns for expandable buttons on LinkedIn profiles.
   * LinkedIn uses many variants — we cover common ones.
   */
  const EXPAND_BUTTON_TEXTS = [
    "show more",
    "see more",
    "show all experiences",
    "show all education",
    "show all licenses & certifications",
    "show all skills",
    "show all recommendations",
    "show all courses",
    "show all projects",
    "show all publications",
    "show all volunteer experiences",
    "show all honors & awards",
    "show all languages",
    "show all organizations",
    // Generic "show all" pattern
  ];

  /**
   * Find all expandable buttons on the page.
   * LinkedIn uses <button> and <a> elements with various class names.
   */
  function findExpandButtons() {
    const buttons = [];
    const allClickable = document.querySelectorAll(
      'button, a[role="button"], [data-control-name], [aria-expanded="false"]'
    );

    for (const el of allClickable) {
      const text = (el.textContent || "").trim().toLowerCase();
      // Check against known expand text patterns
      const isExpand =
        EXPAND_BUTTON_TEXTS.some((t) => text.includes(t)) ||
        text.match(/show \d+ more/i) ||
        text.match(/show all \d+/i) ||
        text.match(/\+\d+ more/i) ||
        text.match(/view \d+ more/i) ||
        text.match(/see \d+ more/i) ||
        text.match(/^view full/i);

      // Also detect icon-only expand buttons via aria-label
      const ariaLabel = (el.getAttribute("aria-label") || "").toLowerCase();
      const isAriaExpand =
        ariaLabel.includes("show more") ||
        ariaLabel.includes("see more") ||
        ariaLabel.includes("expand") ||
        ariaLabel.includes("show all");

      if ((isExpand || isAriaExpand) && el.offsetParent !== null) {
        buttons.push(el);
      }
    }

    // Also look for LinkedIn's inline "see more" within text blocks
    const inlineSeeMore = document.querySelectorAll(
      '.inline-show-more-text__button, .lt-line-clamp__more, [data-control-name="see_more"], .pv-profile-section__see-more-inline'
    );
    for (const el of inlineSeeMore) {
      if (el.offsetParent !== null && !buttons.includes(el)) {
        buttons.push(el);
      }
    }

    return buttons;
  }

  /**
   * Wait for DOM mutations after clicking a button — resolves when no new
   * mutations fire for `quietMs` or after `timeoutMs` total.
   */
  function waitForDomSettle(quietMs = 800, timeoutMs = 8000) {
    return new Promise((resolve) => {
      let timer;
      const observer = new MutationObserver(() => {
        clearTimeout(timer);
        timer = setTimeout(() => {
          observer.disconnect();
          resolve();
        }, quietMs);
      });

      observer.observe(document.body, {
        childList: true,
        subtree: true,
        characterData: true,
      });

      // Start the quiet timer immediately in case no mutations fire
      timer = setTimeout(() => {
        observer.disconnect();
        resolve();
      }, quietMs);

      // Hard timeout
      setTimeout(() => {
        observer.disconnect();
        resolve();
      }, timeoutMs);
    });
  }

  /**
   * Expand all hidden sections. Performs multiple passes because expanding
   * one section can reveal more expand buttons.
   */
  /**
   * Scroll through the entire page to trigger lazy-loaded content.
   * LinkedIn lazy-loads images and sections as you scroll.
   */
  async function scrollFullPage() {
    const scrollStep = window.innerHeight * 0.7;
    const maxScroll = document.documentElement.scrollHeight;

    for (let pos = 0; pos < maxScroll; pos += scrollStep) {
      window.scrollTo(0, pos);
      await sleep(300);
    }
    // Scroll to bottom to catch anything at the end
    window.scrollTo(0, document.documentElement.scrollHeight);
    await sleep(500);
    // Return to top
    window.scrollTo(0, 0);
    await sleep(200);
  }

  async function expandAllSections() {
    // First scroll the entire page to trigger lazy-loading of sections
    await scrollFullPage();

    const MAX_PASSES = 5;
    let totalClicked = 0;

    for (let pass = 0; pass < MAX_PASSES; pass++) {
      const buttons = findExpandButtons();
      if (buttons.length === 0) break;

      for (const btn of buttons) {
        try {
          btn.scrollIntoView({ behavior: "instant", block: "center" });
          await sleep(250);
          btn.click();
          totalClicked++;
          await waitForDomSettle(800, 4000);
          await sleep(500);
        } catch (_) {
          // Button may have been removed from DOM after a prior click
        }
      }
    }

    // After expanding, scroll full page once more to load any newly revealed lazy content
    await scrollFullPage();

    return totalClicked;
  }

  // ---------------------------------------------------------------------------
  // 2. Profile Data Extraction
  // ---------------------------------------------------------------------------

  /** Safely get trimmed text from the first matching element */
  function getText(selector, context = document) {
    const el = context.querySelector(selector);
    return el ? el.textContent.trim() : "";
  }

  /** Get all text blocks from multiple matching elements */
  function getAllText(selector, context = document) {
    return Array.from(context.querySelectorAll(selector))
      .map((el) => el.textContent.trim())
      .filter(Boolean);
  }

  /** Extract the full name */
  function extractName() {
    // Primary: LinkedIn's heading class
    const name = getText(".text-heading-xlarge");
    if (name) return name;
    // Fallback: h1 on the profile
    return getText("h1");
  }

  /** Extract the headline (title / role tagline) */
  function extractHeadline() {
    const hl = getText(".text-body-medium");
    if (hl) return hl;
    // Fallback
    return getText(".pv-top-card--list .text-body-medium");
  }

  /** Extract location */
  function extractLocation() {
    // Location is typically the second .text-body-small in the top card
    const spans = document.querySelectorAll(
      ".pv-top-card--list .text-body-small"
    );
    for (const s of spans) {
      const t = s.textContent.trim();
      // Skip connection count or empty
      if (t && !t.match(/^\d+ connections?$/i) && !t.match(/^\d+ followers?$/i)) {
        return t;
      }
    }
    // Broader fallback
    const loc = getText('span.text-body-small[class*="top-card"]');
    return loc || "";
  }

  /** Extract the About section */
  async function extractAbout() {
    const sections = document.querySelectorAll("section");
    for (const sec of sections) {
      const heading = sec.querySelector("#about");
      if (heading) {
        // Click any "see more" / "show more" button within the About section first
        const seeMoreBtn = sec.querySelector(
          '.inline-show-more-text__button, .lt-line-clamp__more, button[aria-expanded="false"]'
        );
        if (seeMoreBtn && seeMoreBtn.offsetParent !== null) {
          seeMoreBtn.click();
          await sleep(400);
        }

        // The actual text is usually in a div/span after the heading
        const textBlock = sec.querySelector(
          ".pv-shared-text-with-see-more span[aria-hidden='true'], .display-flex span[aria-hidden='true'], .inline-show-more-text span[aria-hidden='true']"
        );
        if (textBlock) return textBlock.textContent.trim();

        // Fallback: gather all visible text spans in the section (skip the heading)
        const spans = sec.querySelectorAll('span[aria-hidden="true"], span.visually-hidden');
        for (const sp of spans) {
          const t = sp.textContent.trim();
          if (t.length > 50) return t;
        }

        // Last resort: get the full section text minus the heading
        const fullText = sec.textContent.replace(/About/i, "").trim();
        if (fullText.length > 50) return fullText;
      }
    }
    return "";
  }

  /** Extract experience entries */
  function extractExperience() {
    const entries = [];
    const sections = document.querySelectorAll("section");

    for (const sec of sections) {
      const heading = sec.querySelector("#experience");
      if (!heading) continue;

      // Each experience item is in a li within the section
      const items = sec.querySelectorAll("li.artdeco-list__item");
      if (items.length === 0) {
        // Fallback: try direct list items
        const fallbackItems = sec.querySelectorAll(
          'ul > li[class*="pvs-list__paged-list-item"]'
        );
        for (const item of fallbackItems) {
          entries.push(parseExperienceItem(item));
        }
      } else {
        for (const item of items) {
          entries.push(parseExperienceItem(item));
        }
      }
      break;
    }

    // Broader fallback using data-testid or aria
    if (entries.length === 0) {
      const expSection = document.querySelector(
        '[id="experience"] ~ .pvs-list__outer-container'
      );
      if (expSection) {
        const items = expSection.querySelectorAll("li");
        for (const item of items) {
          const parsed = parseExperienceItem(item);
          if (parsed.title || parsed.company) entries.push(parsed);
        }
      }
    }

    return entries;
  }

  function parseExperienceItem(li) {
    const visibleSpans = Array.from(
      li.querySelectorAll('span[aria-hidden="true"]')
    ).map((s) => s.textContent.trim()).filter(Boolean);

    // Use heuristics instead of fixed positions — LinkedIn can reorder these
    const datePattern = /\b(jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec|present|\d{4})\b/i;
    const locationPattern = /,\s*(united|canada|uk|australia|india|remote|area|metro)/i;
    const durationPattern = /\b\d+\s*(yr|year|mo|month|mos|yrs)\b/i;

    let title = "";
    let company = "";
    let dates = "";
    let location = "";
    const descParts = [];

    for (const text of visibleSpans) {
      if (!dates && (datePattern.test(text) || durationPattern.test(text))) {
        dates = text;
      } else if (!location && (locationPattern.test(text) || text.match(/^[A-Z][a-z]+,\s*[A-Z]/))) {
        location = text;
      } else if (!title) {
        title = text;
      } else if (!company) {
        company = text;
      } else {
        descParts.push(text);
      }
    }

    return {
      title,
      company,
      dates,
      location,
      description: descParts.join("\n").trim(),
    };
  }

  /** Extract education entries */
  function extractEducation() {
    const entries = [];
    const sections = document.querySelectorAll("section");

    for (const sec of sections) {
      const heading = sec.querySelector("#education");
      if (!heading) continue;

      const items = sec.querySelectorAll(
        'li.artdeco-list__item, li[class*="pvs-list__paged-list-item"]'
      );
      for (const item of items) {
        const visibleSpans = Array.from(
          item.querySelectorAll('span[aria-hidden="true"]')
        ).map((s) => s.textContent.trim()).filter(Boolean);

        const datePattern = /\b(jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec|\d{4})\b/i;
        let school = "";
        let degree = "";
        let dates = "";
        const detailParts = [];

        for (const text of visibleSpans) {
          if (!dates && datePattern.test(text) && text.match(/\d{4}/)) {
            dates = text;
          } else if (!school) {
            school = text;
          } else if (!degree) {
            degree = text;
          } else {
            detailParts.push(text);
          }
        }

        entries.push({
          school,
          degree,
          dates,
          details: detailParts.join("\n").trim(),
        });
      }
      break;
    }

    return entries;
  }

  /** Extract skills */
  function extractSkills() {
    const skills = [];
    const sections = document.querySelectorAll("section");

    for (const sec of sections) {
      const heading = sec.querySelector("#skills");
      if (!heading) continue;

      const items = sec.querySelectorAll(
        'li.artdeco-list__item, li[class*="pvs-list__paged-list-item"]'
      );
      for (const item of items) {
        const span = item.querySelector('span[aria-hidden="true"]');
        if (span) {
          const text = span.textContent.trim();
          if (text && !text.match(/^\d+$/)) {
            skills.push(text);
          }
        }
      }
      break;
    }

    return skills;
  }

  /** Extract certifications */
  function extractCertifications() {
    const certs = [];
    const sections = document.querySelectorAll("section");

    for (const sec of sections) {
      const heading = sec.querySelector(
        "#licenses_and_certifications, #certifications"
      );
      if (!heading) continue;

      const items = sec.querySelectorAll(
        'li.artdeco-list__item, li[class*="pvs-list__paged-list-item"]'
      );
      for (const item of items) {
        const visibleSpans = Array.from(
          item.querySelectorAll('span[aria-hidden="true"]')
        ).map((s) => s.textContent.trim());

        certs.push({
          name: visibleSpans[0] || "",
          issuer: visibleSpans[1] || "",
          date: visibleSpans[2] || "",
        });
      }
      break;
    }

    return certs;
  }

  /** Extract recommendations count */
  function extractRecommendationsCount() {
    const sections = document.querySelectorAll("section");
    for (const sec of sections) {
      const heading = sec.querySelector("#recommendations");
      if (!heading) continue;
      const items = sec.querySelectorAll(
        'li.artdeco-list__item, li[class*="pvs-list__paged-list-item"]'
      );
      return items.length;
    }

    // Fallback: look for tab buttons that say "Received (X)"
    const tabs = document.querySelectorAll("button");
    for (const tab of tabs) {
      const match = tab.textContent.match(/Received\s*\((\d+)\)/i);
      if (match) return parseInt(match[1], 10);
    }

    return 0;
  }

  /** Build a readable raw_text string from structured profile data */
  function buildRawText(name, headline, location, about, experience, education, skills, certs, recsCount) {
    const parts = [];
    parts.push(`${name} - ${headline}`);
    if (location) parts.push(`Location: ${location}`);
    if (about) parts.push(`\nAbout:\n${about}`);

    if (experience.length > 0) {
      parts.push("\nExperience:");
      for (const exp of experience) {
        let line = `  ${exp.title}`;
        if (exp.company) line += ` at ${exp.company}`;
        if (exp.dates) line += ` (${exp.dates})`;
        if (exp.location) line += ` - ${exp.location}`;
        parts.push(line);
        if (exp.description) parts.push(`    ${exp.description}`);
      }
    }

    if (education.length > 0) {
      parts.push("\nEducation:");
      for (const edu of education) {
        let line = `  ${edu.school}`;
        if (edu.degree) line += ` - ${edu.degree}`;
        if (edu.dates) line += ` (${edu.dates})`;
        parts.push(line);
        if (edu.details) parts.push(`    ${edu.details}`);
      }
    }

    if (skills.length > 0) {
      parts.push(`\nSkills: ${skills.join(", ")}`);
    }

    if (certs.length > 0) {
      parts.push("\nCertifications:");
      for (const c of certs) {
        let line = `  ${c.name}`;
        if (c.issuer) line += ` - ${c.issuer}`;
        if (c.date) line += ` (${c.date})`;
        parts.push(line);
      }
    }

    if (recsCount > 0) {
      parts.push(`\nRecommendations: ${recsCount}`);
    }

    return parts.join("\n");
  }

  /** Assemble the full profile data object matching the API schema */
  async function extractProfileData() {
    const full_name = extractName();
    const headline = extractHeadline();
    const location = extractLocation();
    const about = await extractAbout();
    const experience = extractExperience();
    const education = extractEducation();
    const skills = extractSkills();
    const certs = extractCertifications();
    const recsCount = extractRecommendationsCount();

    // Derive company from first experience entry
    const company = experience.length > 0 ? experience[0].company : "";

    const raw_text = buildRawText(full_name, headline, location, about, experience, education, skills, certs, recsCount);

    const structured = { about, experience, education, skills, certifications: certs, recommendations_count: recsCount };

    const data = {
      linkedin_url: window.location.href.split("?")[0],
      extracted_at: new Date().toISOString(),
      full_name,
      headline,
      company,
      location,
      raw_text,
      structured_data: JSON.stringify(structured),
    };

    return data;
  }

  // ---------------------------------------------------------------------------
  // 3. Message Listener (triggered by popup)
  // ---------------------------------------------------------------------------

  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.action !== "extract") return false;

    // Verify we're on a LinkedIn profile page
    if (!window.location.href.match(/linkedin\.com\/in\//)) {
      sendResponse({
        success: false,
        error: "Not on a LinkedIn profile page",
      });
      return true;
    }

    // Run the async extraction pipeline
    (async () => {
      try {
        // Step 1: Expand all hidden sections
        reportProgress("expanding", "Expanding hidden sections...");
        const clicked = await expandAllSections();
        reportProgress(
          "expanding_done",
          `Expanded ${clicked} section${clicked !== 1 ? "s" : ""}`
        );

        // Step 2: Extract structured data
        reportProgress("extracting", "Extracting profile text...");
        await sleep(300); // Brief pause after final expansions
        const profileData = await extractProfileData();
        reportProgress("extracting_done", `Extracted: ${profileData.full_name}`);

        // Step 3: Send to background for API submission
        reportProgress("sending", "Sending to server...");
        chrome.runtime.sendMessage(
          { type: "profile_extracted", data: profileData },
          (apiResponse) => {
            if (chrome.runtime.lastError) {
              sendResponse({
                success: false,
                error: "Failed to communicate with background worker",
              });
              return;
            }

            if (apiResponse && apiResponse.success) {
              reportProgress("done", "Profile saved successfully");
              sendResponse({
                success: true,
                name: profileData.full_name,
                data: profileData,
              });
            } else {
              reportProgress(
                "error",
                apiResponse?.error || "Server rejected the profile"
              );
              sendResponse({
                success: false,
                error: apiResponse?.error || "API submission failed",
                data: profileData, // Still return extracted data
              });
            }
          }
        );
      } catch (err) {
        reportProgress("error", err.message);
        sendResponse({ success: false, error: err.message });
      }
    })();

    return true; // Keep message channel open for async sendResponse
  });
})();
