/**
 * LinkReader - Extension Popup
 * Manages auth state (login form vs main view), triggers extraction,
 * shows step-by-step progress, and lists recently extracted profiles.
 * JWT stored in chrome.storage.local — never localStorage.
 */

(() => {
  "use strict";

  // ---- DOM references ----
  const loginSection = document.getElementById("loginSection");
  const mainSection = document.getElementById("mainSection");
  const loginBtn = document.getElementById("loginBtn");
  const logoutBtn = document.getElementById("logoutBtn");
  const emailInput = document.getElementById("emailInput");
  const passwordInput = document.getElementById("passwordInput");
  const loginError = document.getElementById("loginError");
  const userAvatar = document.getElementById("userAvatar");
  const userName = document.getElementById("userName");
  const usageBadge = document.getElementById("usageBadge");
  const statusEl = document.getElementById("status");
  const extractBtn = document.getElementById("extractBtn");
  const progressSteps = document.getElementById("progressSteps");
  const recentList = document.getElementById("recentList");

  const steps = {
    expand: document.getElementById("stepExpand"),
    extract: document.getElementById("stepExtract"),
    send: document.getElementById("stepSend"),
    done: document.getElementById("stepDone"),
  };

  // ---- Helpers ----

  function getInitials(name) {
    if (!name) return "?";
    return name
      .split(" ")
      .map((w) => w[0])
      .join("")
      .toUpperCase()
      .slice(0, 2);
  }

  function showLogin() {
    loginSection.classList.add("active");
    mainSection.classList.remove("active");
    logoutBtn.style.display = "none";
  }

  function showMain(user) {
    loginSection.classList.remove("active");
    mainSection.classList.add("active");
    logoutBtn.style.display = "flex";

    if (user) {
      userAvatar.textContent = getInitials(user.name || user.email);
      userName.textContent = user.name || user.email;
    }
  }

  function showLoginError(msg) {
    loginError.textContent = msg;
    loginError.classList.add("visible");
  }

  function clearLoginError() {
    loginError.textContent = "";
    loginError.classList.remove("visible");
  }

  function setStatus(text, type) {
    statusEl.textContent = text;
    statusEl.className = `status ${type}`;
  }

  // ---- Progress step management ----

  function resetProgress() {
    progressSteps.classList.remove("active");
    for (const step of Object.values(steps)) {
      step.className = "step";
    }
    // Reset icons to numbers
    document.getElementById("stepExpandIcon").innerHTML = "1";
    document.getElementById("stepExtractIcon").innerHTML = "2";
    document.getElementById("stepSendIcon").innerHTML = "3";
    document.getElementById("stepDoneIcon").innerHTML = "4";
  }

  function showProgress() {
    progressSteps.classList.add("active");
  }

  function setStepState(stepKey, state) {
    const step = steps[stepKey];
    if (!step) return;

    step.className = `step ${state}`;
    const iconEl = step.querySelector(".step-icon");

    if (state === "active") {
      iconEl.innerHTML = '<div class="spinner"></div>';
    } else if (state === "done") {
      iconEl.innerHTML = "&#x2713;"; // checkmark
    } else if (state === "error") {
      iconEl.innerHTML = "&#x2717;"; // X mark
    }
  }

  // ---- Recent profiles rendering ----

  function renderRecentProfiles(profiles) {
    if (!profiles || profiles.length === 0) {
      recentList.innerHTML =
        '<div class="empty-state">No profiles extracted yet</div>';
      return;
    }

    recentList.innerHTML = profiles
      .map(
        (p) => `
      <div class="recent-item">
        <div class="recent-avatar">${getInitials(p.name)}</div>
        <div class="recent-info">
          <div class="recent-name">${escapeHtml(p.name || "Unknown")}</div>
          <div class="recent-headline">${escapeHtml(p.headline || "")}</div>
        </div>
      </div>
    `
      )
      .join("");
  }

  function escapeHtml(str) {
    const div = document.createElement("div");
    div.textContent = str;
    return div.innerHTML;
  }

  function loadRecentProfiles() {
    chrome.runtime.sendMessage({ type: "get_recent" }, (resp) => {
      if (resp && resp.profiles) {
        renderRecentProfiles(resp.profiles);
      }
    });
  }

  // ---- Usage display ----

  function loadUsage() {
    chrome.runtime.sendMessage({ type: "get_usage" }, (resp) => {
      if (resp && resp.success && resp.data) {
        const used = resp.data.used || 0;
        const limit = resp.data.limit || 100;
        usageBadge.textContent = `${used}/${limit} used`;
      } else {
        usageBadge.textContent = "--/--";
      }
    });
  }

  // ---- Auth check on popup open ----

  function checkAuth() {
    chrome.runtime.sendMessage({ type: "check_auth" }, (resp) => {
      if (resp && resp.loggedIn) {
        showMain(resp.user);
        loadUsage();
        loadRecentProfiles();
        checkLinkedInPage();
      } else {
        showLogin();
      }
    });
  }

  // ---- LinkedIn page detection ----

  function checkLinkedInPage() {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      const tab = tabs[0];
      if (tab && tab.url && tab.url.match(/linkedin\.com\/in\//)) {
        setStatus("LinkedIn profile detected", "ready");
        extractBtn.disabled = false;
      } else {
        setStatus("Navigate to a LinkedIn profile to extract", "not-linkedin");
        extractBtn.disabled = true;
      }
    });
  }

  // ---- Login handler ----

  loginBtn.addEventListener("click", () => {
    const email = emailInput.value.trim();
    const password = passwordInput.value;

    if (!email || !password) {
      showLoginError("Please enter email and password");
      return;
    }

    clearLoginError();
    loginBtn.disabled = true;
    loginBtn.textContent = "Signing in...";

    chrome.runtime.sendMessage(
      { type: "login", email, password },
      (resp) => {
        loginBtn.disabled = false;
        loginBtn.textContent = "Sign In";

        if (resp && resp.success) {
          showMain(resp.user);
          loadUsage();
          loadRecentProfiles();
          checkLinkedInPage();
        } else {
          showLoginError(resp?.error || "Login failed");
        }
      }
    );
  });

  // Enter key submits login
  passwordInput.addEventListener("keydown", (e) => {
    if (e.key === "Enter") loginBtn.click();
  });
  emailInput.addEventListener("keydown", (e) => {
    if (e.key === "Enter") passwordInput.focus();
  });

  // ---- Logout handler ----

  logoutBtn.addEventListener("click", () => {
    chrome.runtime.sendMessage({ type: "logout" }, () => {
      showLogin();
      emailInput.value = "";
      passwordInput.value = "";
      clearLoginError();
    });
  });

  // ---- Extract button handler ----

  extractBtn.addEventListener("click", async () => {
    extractBtn.disabled = true;
    extractBtn.textContent = "Extracting...";
    resetProgress();
    showProgress();
    setStepState("expand", "active");
    setStatus("Extraction in progress...", "ready");

    try {
      const [tab] = await chrome.tabs.query({
        active: true,
        currentWindow: true,
      });

      // Ensure content script is injected before sending message.
      // This prevents the race condition where the script hasn't loaded yet.
      try {
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          files: ["content.js"],
        });
      } catch (_injectionErr) {
        // Script may already be injected or page may block injection — that's OK.
        // The sendMessage below will still work if the script was auto-injected.
      }

      // Brief pause to let the content script's message listener register
      await new Promise((r) => setTimeout(r, 150));

      const response = await chrome.tabs.sendMessage(tab.id, {
        action: "extract",
      });

      if (response && response.success) {
        setStepState("done", "done");
        setStatus(`Extracted: ${response.name || "Profile"}`, "success");
        extractBtn.textContent = "Extracted!";
        loadUsage();
        loadRecentProfiles();
      } else {
        setStatus(response?.error || "Extraction failed", "not-linkedin");
        // Mark current step as error
        for (const key of ["done", "send", "extract", "expand"]) {
          if (steps[key].classList.contains("active")) {
            setStepState(key, "error");
            break;
          }
        }
        extractBtn.textContent = "Retry";
        extractBtn.disabled = false;
      }
    } catch (err) {
      const msg = err.message || "";
      if (msg.includes("Receiving end does not exist")) {
        setStatus("Page not ready — please refresh LinkedIn and try again", "not-linkedin");
      } else {
        setStatus(`Error: ${msg || "Could not connect to page"}`, "not-linkedin");
      }
      setStepState("expand", "error");
      extractBtn.textContent = "Retry";
      extractBtn.disabled = false;
    }
  });

  // ---- Listen for progress updates from content script (via background) ----

  chrome.runtime.onMessage.addListener((message) => {
    if (message.type !== "extraction_progress") return;

    const { step } = message;

    if (step === "expanding") {
      setStepState("expand", "active");
    } else if (step === "expanding_done") {
      setStepState("expand", "done");
    } else if (step === "extracting") {
      setStepState("extract", "active");
    } else if (step === "extracting_done") {
      setStepState("extract", "done");
    } else if (step === "sending") {
      setStepState("send", "active");
    } else if (step === "done") {
      setStepState("send", "done");
      setStepState("done", "done");
    } else if (step === "error") {
      // Find the currently active step and mark it as error
      for (const key of ["expand", "extract", "send", "done"]) {
        if (steps[key].classList.contains("active")) {
          setStepState(key, "error");
          break;
        }
      }
    }
  });

  // ---- Init ----
  checkAuth();
})();
